class Game {
    constructor() {
        this.canvas = document.getElementById('gameCanvas');
        this.ctx = this.canvas.getContext('2d');
        this.resizeCanvas();
        
        // Game state
        this.score = 0;
        this.lives = 3;
        this.gameOver = false;
        this.started = false;
        
        // Player
        this.player = {
            x: 100,
            y: this.canvas.height / 2,
            width: 40,
            height: 20,
            speed: 5
        };
        
        // Obstacles
        this.obstacles = [];
        this.obstacleSpeed = 3;
        this.obstacleSpawnRate = 120;
        this.frameCount = 0;
        
        // Controls
        this.keys = {};
        this.bindEvents();
        
        // Load assets
        this.loadAssets();
    }
    
    loadAssets() {
        // Create airplane SVG - now facing forward
        this.airplane = new Path2D();
        this.airplane.moveTo(40, 10);
        this.airplane.lineTo(0, 0);
        this.airplane.lineTo(0, 20);
        this.airplane.lineTo(40, 10);
        
        // Create loop SVG - unchanged
        this.loop = new Path2D();
        this.loop.arc(0, 0, 30, 0, Math.PI * 2);
    }
    
    bindEvents() {
        window.addEventListener('keydown', (e) => this.keys[e.key] = true);
        window.addEventListener('keyup', (e) => this.keys[e.key] = false);
        window.addEventListener('resize', () => this.resizeCanvas());
        
        document.getElementById('startButton').addEventListener('click', () => this.start());
        document.getElementById('restartButton').addEventListener('click', () => this.restart());
    }
    
    resizeCanvas() {
        this.canvas.width = this.canvas.offsetWidth;
        this.canvas.height = this.canvas.offsetHeight;
    }
    
    start() {
        if (!this.started) {
            this.started = true;
            this.gameLoop();
            document.getElementById('startButton').classList.add('d-none');
        }
    }
    
    restart() {
        this.score = 0;
        this.lives = 3;
        this.gameOver = false;
        this.obstacles = [];
        this.player.y = this.canvas.height / 2;
        document.getElementById('restartButton').classList.add('d-none');
        this.start();
    }
    
    update() {
        if (this.gameOver) return;
        
        // Player movement
        if (this.keys['ArrowUp'] || this.keys['w']) this.player.y -= this.player.speed;
        if (this.keys['ArrowDown'] || this.keys['s']) this.player.y += this.player.speed;
        
        // Keep player in bounds
        this.player.y = Math.max(0, Math.min(this.canvas.height - this.player.height, this.player.y));
        
        // Spawn obstacles
        if (++this.frameCount % this.obstacleSpawnRate === 0) {
            this.obstacles.push({
                x: this.canvas.width,
                y: Math.random() * (this.canvas.height - 100),
                passed: false
            });
        }
        
        // Update obstacles
        this.obstacles.forEach((obstacle, index) => {
            obstacle.x -= this.obstacleSpeed;
            
            // Check collision
            if (this.checkCollision(obstacle)) {
                this.lives--;
                this.obstacles.splice(index, 1);
                if (this.lives <= 0) this.gameOver = true;
            }
            
            // Score points
            if (!obstacle.passed && obstacle.x < this.player.x) {
                this.score += 10;
                obstacle.passed = true;
            }
            
            // Remove off-screen obstacles
            if (obstacle.x + 60 < 0) {
                this.obstacles.splice(index, 1);
            }
        });
        
        // Update UI
        document.getElementById('score').textContent = this.score;
        document.getElementById('lives').textContent = this.lives;
        
        if (this.gameOver) {
            document.getElementById('restartButton').classList.remove('d-none');
            this.saveScore();
        }
    }
    
    checkCollision(obstacle) {
        return (
            this.player.x < obstacle.x + 60 &&
            this.player.x + this.player.width > obstacle.x &&
            this.player.y < obstacle.y + 60 &&
            this.player.y + this.player.height > obstacle.y
        );
    }
    
    draw() {
        // Clear canvas
        this.ctx.fillStyle = '#1a1a1a';
        this.ctx.fillRect(0, 0, this.canvas.width, this.canvas.height);
        
        // Draw player
        this.ctx.save();
        this.ctx.translate(this.player.x, this.player.y);
        this.ctx.fillStyle = '#ffffff';
        this.ctx.fill(this.airplane);
        this.ctx.restore();
        
        // Draw obstacles
        this.obstacles.forEach(obstacle => {
            this.ctx.save();
            this.ctx.translate(obstacle.x, obstacle.y);
            this.ctx.fillStyle = '#ff0000';
            this.ctx.fill(this.loop);
            this.ctx.restore();
        });
        
        if (this.gameOver) {
            this.ctx.fillStyle = '#ffffff';
            this.ctx.font = '48px Arial';
            this.ctx.fillText('Game Over!', this.canvas.width/2 - 100, this.canvas.height/2);
        }
    }
    
    async saveScore() {
        try {
            await fetch('/api/scores', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    score: this.score,
                    timestamp: new Date().toISOString()
                })
            });
            this.updateHighScores();
        } catch (error) {
            console.error('Error saving score:', error);
        }
    }
    
    async updateHighScores() {
        try {
            const response = await fetch('/api/scores');
            const scores = await response.json();
            const scoresList = document.getElementById('highScoresList');
            scoresList.innerHTML = scores
                .map(score => `<li>${score.score} points</li>`)
                .join('');
        } catch (error) {
            console.error('Error updating high scores:', error);
        }
    }
    
    gameLoop() {
        if (!this.gameOver) {
            this.update();
            this.draw();
            requestAnimationFrame(() => this.gameLoop());
        }
    }
}

// Start game when page loads
window.addEventListener('load', () => {
    const game = new Game();
});